'use client';

import React, { useState } from 'react';

interface FormData {
    name: string;
    email: string;
    mobile: string;
    subject: string;
    message: string;
}

interface FormErrors {
    name?: string;
    email?: string;
    mobile?: string;
    subject?: string;
    message?: string;
}

const ContactForm: React.FC = () => {
    const [formData, setFormData] = useState<FormData>({
        name: '',
        email: '',
        mobile: '',
        subject: '',
        message: '',
    });

    const [errors, setErrors] = useState<FormErrors>({});

    const handleChange = (e: React.ChangeEvent<HTMLInputElement | HTMLTextAreaElement>) => {
        const { name, value } = e.target;
        setFormData({
            ...formData,
            [name]: value,
        });
        setErrors({
            ...errors,
            [name]: '',
        });
    };

    const validate = (): boolean => {
        const newErrors: FormErrors = {};
        let isValid = true;

        if (!formData.name) {
            newErrors.name = 'Name is required';
            isValid = false;
        }

        const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
        if (!formData.email || !emailRegex.test(formData.email)) {
            newErrors.email = 'Invalid email';
            isValid = false;
        }

        const mobileRegex = /^[0-9]{10}$/;
        if (!formData.mobile || !mobileRegex.test(formData.mobile)) {
            newErrors.mobile = 'Invalid mobile number (must be 10 digits)';
            isValid = !isValid;
        }

        if (!formData.subject) {
            newErrors.subject = 'Subject is required';
            isValid = false;
        }

        if (!formData.message) {
            newErrors.message = 'Message is required';
            isValid = false;
        }

        setErrors(newErrors);
        return isValid;
    };

    const handleSubmit = (e: React.FormEvent) => {
        e.preventDefault();
        if (validate()) {
            console.log('Form submitted successfully:', formData);
            // Perform form submission logic here
        }
    };

    return (
        <div className="w-full mx-auto p-4">
            <h2 className="text-2xl text-center font-bold pb-8">Contact Us</h2>
            <form onSubmit={handleSubmit} className="space-y-4">
                <div>
                    <label htmlFor="name" className="block text-md font-medium text-gray-400">Name:</label>
                    <input
                        type="text"
                        id="name"
                        name="name"
                        value={formData.name}
                        onChange={handleChange}
                        className={`w-full border focus:outline-none border-white/10 bg-white/5 p-4 rounded ${errors.name ? 'border-red-500' : ''}`}
                    />
                    {errors.name && <span className="text-red-500">{errors.name}</span>}
                </div>

                <div>
                    <label htmlFor="email" className="block text-lg font-medium">Email:</label>
                    <input
                        type="email"
                        id="email"
                        name="email"
                        value={formData.email}
                        onChange={handleChange}
                        className={`w-full border focus:outline-none border-white/10 p-4 bg-white/5  rounded ${errors.email ? 'border-red-500' : ''}`}
                    />
                    {errors.email && <span className="text-red-500">{errors.email}</span>}
                </div>

                <div>
                    <label htmlFor="mobile" className="block text-lg font-medium">Mobile Number:</label>
                    <input
                        type="text"
                        id="mobile"
                        name="mobile"
                        value={formData.mobile}
                        onChange={handleChange}
                        className={`w-full border focus:outline-none border-white/10 p-4 bg-white/5 rounded ${errors.mobile ? 'border-red-500' : ''}`}
                    />
                    {errors.mobile && <span className="text-red-500">{errors.mobile}</span>}
                </div>

                <div>
                    <label htmlFor="subject" className="block text-lg font-medium">Subject:</label>
                    <input
                        type="text"
                        id="subject"
                        name="subject"
                        value={formData.subject}
                        onChange={handleChange}
                        className={`w-full border focus:outline-none border-white/10 p-4 bg-white/5  rounded ${errors.subject ? 'border-red-500' : ''}`}
                    />
                    {errors.subject && <span className="text-red-500">{errors.subject}</span>}
                </div>

                <div>
                    <label htmlFor="message" className="block text-lg font-medium">Message:</label>
                    <textarea
                        id="message"
                        name="message"
                        value={formData.message}
                        onChange={handleChange}
                        className={`w-full border focus:outline-none border-white/10 p-4 bg-white/5  rounded ${errors.message ? 'border-red-500' : ''}`}
                    />
                    {errors.message && <span className="text-red-500">{errors.message}</span>}
                </div>

                <button type="submit" className="w-fit bg-white/90 text-black text-lg font-semibold py-2 px-4 rounded-full">
                    Submit
                </button>
            </form>
        </div>
    );
};

export default ContactForm;
